<?php
/**
 * Copyright 2017 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsd.
 *
 * @author   Jan Schneider <jan@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/bsd BSD
 * @package  Refactor
 */

namespace Horde\Refactor;

use Horde\Refactor\Rule;
use Horde\Refactor\Exception\StopProcessing;
use Horde_Argv_Parser;
use Horde_Argv_Option;
use Horde_Cli;

/**
 * Command line tool for refactoring of PHP code.
 *
 * @author    Jan Schneider <jan@horde.org>
 * @category  Horde
 * @copyright 2017 Horde LLC
 * @license   http://www.horde.org/licenses/bsd BSD
 * @package   Refactor
 */
class Cli
{
    /**
     * The main entry point for the application.
     */
    public static function main()
    {
        $parser = new Horde_Argv_Parser(
            array('usage' => '%prog [OPTIONS] RefactoringFile.php RefactoringClass')
        );
        $parser->addOptions(
            array(
                new Horde_Argv_Option(
                    '-f',
                    '--file',
                    array(
                        'action' => 'store',
                        'help'   => 'File to be refactored',
                    )
                ),
                new Horde_Argv_Option(
                    '-d',
                    '--directory',
                    array(
                        'action' => 'store',
                        'help'   => 'Directory to be recursively refactored',
                    )
                ),
                new Horde_Argv_Option(
                    '-u',
                    '--update',
                    array(
                        'action' => 'store_true',
                        'help'   => 'Overwrite the refatored files',
                    )
                ),
                new Horde_Argv_Option(
                    '-c',
                    '--config-file',
                    array(
                        'action' => 'store',
                        'help'   => 'Configuration file to use',
                    )
                ),
            )
        );
        list($options, $arguments) = $parser->parseArgs();

        if ((!$options->file && !$options->directory) ||
            count($arguments) != 2) {
            $parser->printHelp();
            return;
        }

        $cli = Horde_Cli::init();
        if (!$options->update) {
            $renderer = new \Horde_Text_Diff_Renderer_Unified_Colored(
                array('cli' => $cli)
            );
        }
        if ($options->file) {
            $files = array($options->file);
            $header = 'Processing single file ' . $options->file;
        } else {
            $header = 'Processing directory ' . $options->directory;
            $files = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator(
                    $options->directory,
                    \FilesystemIterator::CURRENT_AS_PATHNAME
                    | \FilesystemIterator::SKIP_DOTS
                ),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );
        }
        $cli->writeln($cli->bold($cli->header($header, '=', '=')));

        require $arguments[0];
        $config = new Config($options->config_file);
        $class = 'Horde\\Refactor\\Rule\\' . $arguments[1];

        foreach ($files as $file) {
            if (substr($file, -4) != '.php') {
                continue;
            }
            $cli->writeln();
            $cli->writeln(
                $cli->header(
                    "Processing file\n" . $cli->color('cyan', $file),
                    '-', '-'
                )
            );
            $rule = new $class($file, $config->{$arguments[1]});
            try {
                $rule->run();
            } catch (StopProcessing $e) {
                $cli->writeln($cli->color('red', $e->getMessage()));
                continue;
            }
            if ($rule->warnings) {
                $cli->writeln();
                foreach ($rule->warnings as $warning) {
                    $cli->writeln($cli->color('brown', $warning));
                }
            }
            $original = file($file, FILE_IGNORE_NEW_LINES);
            $refactored = explode("\n", trim($rule->dump()));
            if (!array_diff($original, $refactored) &&
                !array_diff($refactored, $original)) {
                $cli->writeln($cli->blue('Refactoring not necessary'));
                continue;
            }
            if ($options->update) {
                file_put_contents($file, $rule->dump());
                $cli->writeln($cli->green('Updated file'));
            } else {
                $diff = new \Horde_Text_Diff(
                    'auto',
                    array($original, $refactored)
                );
                $cli->writeln();
                echo $renderer->render($diff);
            }
        }
    }
}