<?php
/**
 * Horde_Pear_Package_Type_Horde:: deals with packages provided by Horde.
 *
 * PHP version 5
 *
 * @category Horde
 * @package  Pear
 * @author   Michael J Rubinsky <mrubinsk@horde.org>
 * @author   Gunnar Wrobel <wrobel@pardus.de>
 * @license  http://www.horde.org/licenses/lgpl21 LGPL 2.1
 */

/**
 * Deals with packages provided by Horde in the split repository structure.
 *
 * Copyright 2011-2016 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (LGPL). If you
 * did not receive this file, see http://www.horde.org/licenses/lgpl21.
 *
 * @category Horde
 * @package  Pear
 * @author   Michael J Rubinsky <mrubinsk@horde.org>
 * @author   Gunnar Wrobel <wrobel@pardus.de>
 * @license  http://www.horde.org/licenses/lgpl21 LGPL 2.1
 */
class Horde_Pear_Package_Type_HordeSplit extends Horde_Pear_Package_Type_Horde
{
    /**
     * Get the package type.
     *
     * @return string The type: "Application" or "Component".
     * @throws  Horde_Pear_Exception
     */
    public function getType()
    {
        if (!file_exists($this->getRootPath() . '/.horde.yml')) {
            throw new Horde_Pear_Exception('Cannot find the horde.yml file.');
        }
        $yml = Horde_Yaml::loadFile($this->getRootPath() . '/.horde.yml');
        if (!isset($yml['type'])) {
            throw new Horde_Pear_Exception('Missing type in ' . $this->getRootPath() . '/.horde.yml');
        }
        return ($yml['type'] == 'application') ? 'Application' : 'Component';
    }

    /**
     * Get the package name.
     *
     * @return string The package name.
     */
    public function getName()
    {
        if ($this->getType() == 'Application') {
            return basename($this->getRootPath());
        } else {
            return 'Horde_' . basename($this->getRootPath());
        }
    }

    /**
     * Return the ignore handler for this package.
     *
     * @return Horde_Pear_Package_Contents_Ignore The ignore handler.
     */
    public function getIgnore()
    {
        return new Horde_Pear_Package_Contents_Ignore_Composite(
            array(
                new Horde_Pear_Package_Contents_Ignore_Dot(),
                new Horde_Pear_Package_Contents_Ignore_Symlink(),
                new Horde_Pear_Package_Contents_Ignore_Patterns(
                    array(
                        '*~',
                        'changelog.yml',
                        'conf.php',
                        'CVS/*',
                        '/bin/.htaccess',
                        '/composer.*',
                        '/package.xml',
                        '/phpunit.xml.dist',
                        '/.git/*',
                        '/.gitattributes',
                        '/.gitignore',
                        '/.horde.yml',
                        '/.travis.ini',
                        '/.travis.php7.ini',
                        '/.travis.yml',
                    ),
                    $this->getRepositoryRoot()
                ),
                new Horde_Pear_Package_Contents_Ignore_Git(
                    $this->getGitIgnore(),
                    $this->getRepositoryRoot()
                ),
                new Horde_Pear_Package_Contents_Ignore_Composer()
            )
        );
    }

    /**
     * Return the contents of the Horde .gitignore file.
     *
     * @return string The .gitignore patterns.
     */
    public function getGitIgnore()
    {
        return file_exists($this->getRepositoryRoot() . '/.gitignore')
            ? file_get_contents($this->getRepositoryRoot() . '/.gitignore')
            : '';
    }

    /**
     * The repository root is the same as the package root.
     *
     * @return string The repository path.
     */
    public function getRepositoryRoot()
    {
        return $this->getRootPath();
    }

}
