<?php
/**
 * Copyright 2003-2017 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (ASL).  If you
 * did not receive this file, see http://www.horde.org/licenses/apache.
 *
 * @author   Brent J. Nordquist <bjn@horde.org>
 * @author   Ben Chavet <ben@horde.org>
 * @author   Jan Schneider <jan@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/apache ASL
 * @package  Ingo
 */

/**
 * The Ingo_Script_Procmail class represents a Procmail script generator.
 *
 * @author   Brent J. Nordquist <bjn@horde.org>
 * @author   Ben Chavet <ben@horde.org>
 * @author   Jan Schneider <jan@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/apache ASL
 * @package  Ingo
 */
class Ingo_Script_Procmail extends Ingo_Script_Base
{
    /**
     * A list of driver features.
     *
     * @var array
     */
    protected $_features = array(
        /* Can tests be case sensitive? */
        'case_sensitive' => true,
        /* Does the driver support setting IMAP flags? */
        'imap_flags' => false,
        /* Can this driver perform on demand filtering? */
        'on_demand' => false,
        /* Does the driver require a script file to be generated? */
        'script_file' => true,
        /* Does the driver support the stop-script option? */
        'stop_script' => false,
        /* Does the driver support vacation start and end on time level? */
        'vacation_time' => true,
    );

    /**
     * The list of actions allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_actions = array(
        'Ingo_Rule_User_Discard',
        'Ingo_Rule_User_Keep',
        'Ingo_Rule_User_Move',
        'Ingo_Rule_User_Redirect',
        'Ingo_Rule_User_RedirectKeep',
        'Ingo_Rule_User_Reject'
    );

    /**
     * The categories of filtering allowed.
     *
     * @var array
     */
    protected $_categories = array(
        'Ingo_Rule_System_Blacklist',
        'Ingo_Rule_System_Forward',
        'Ingo_Rule_System_Vacation',
        'Ingo_Rule_System_Forward'
    );

    /**
     * The types of tests allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_types = array(
        Ingo_Rule_User::TEST_HEADER,
        Ingo_Rule_User::TEST_BODY
    );

    /**
     * A list of any special types that this driver supports.
     *
     * @var array
     */
    protected $_special_types = array(
        'Destination'
    );

    /**
     * The list of tests allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_tests = array(
        'contains',
        'not contain',
        'begins with',
        'not begins with',
        'ends with',
        'not ends with',
        'regex'
    );

    /**
     * Constructor.
     *
     * @param array $params  A hash containing parameters needed.
     */
    public function __construct($params = array())
    {
        parent::__construct($params);

        // Bug #10113: Need to explicitly define these variables instead of
        // relying on system defaults.
        if ($this->_params['path_style'] == 'maildir') {
            if (!isset($this->_params['variables']['DEFAULT'])) {
                $this->_params['variables']['DEFAULT'] = '$HOME/Maildir/';
            }
            if (!isset($this->_params['variables']['MAILDIR'])) {
                $this->_params['variables']['MAILDIR'] = '$HOME/Maildir';
            }
        }
    }

    /**
     * Generates the procmail scripts to do the filtering specified in the
     * rules.
     */
    protected function _generate()
    {
        $this->_addItem(
            Ingo::RULE_ALL,
            new Ingo_Script_Procmail_Comment(_("procmail script generated by Ingo") . ' (' . date('F j, Y, g:i a') . ')')
        );

        if (isset($this->_params['forward_file']) &&
            isset($this->_params['forward_string'])) {
            $this->_addItem(
                Ingo::RULE_ALL,
                new Ingo_Script_String($this->_params['forward_string']),
                $this->_params['forward_file']
            );
        }

        /* Add variable information, if present. */
        if (!empty($this->_params['variables']) &&
            is_array($this->_params['variables'])) {
            foreach ($this->_params['variables'] as $key => $val) {
                $this->_addItem(Ingo::RULE_ALL, new Ingo_Script_Procmail_Variable(array('name' => $key, 'value' => $val)));
            }
        }

        $filters = Ingo_Storage_FilterIterator_Skip::create(
            $this->_params['storage'],
            $this->_params['skip']
        );

        foreach ($filters as $rule) {
            switch ($class = get_class($rule)) {
            case 'Ingo_Rule_System_Blacklist':
                $this->_generateBlacklist($rule);
                break;

            case 'Ingo_Rule_System_Forward':
                $this->_generateForward($rule);
                break;

            case 'Ingo_Rule_System_Vacation':
                $this->_generateVacation($rule);
                break;

            case 'Ingo_Rule_System_Whitelist':
                $this->_generateWhitelist($rule);
                break;

            default:
                if (!in_array($class, $this->_actions)) {
                    break;
                }

                /* Create filter if using AND. */
                switch ($rule->combine) {
                case Ingo_Rule_User::COMBINE_ALL:
                    $recipe = new Ingo_Script_Procmail_Recipe(
                        array(
                            'action' => $class,
                            'action-value' => $rule->value,
                            'disable' => $rule->disable
                        ),
                        $this->_params
                    );
                    foreach ($rule->conditions as $condition) {
                        $recipe->addCondition($condition);
                    }
                    $this->_addItem(
                        Ingo::RULE_FILTER,
                        new Ingo_Script_Procmail_Comment($rule->name, $rule->disable, true)
                    );
                    $this->_addItem(Ingo::RULE_FILTER, $recipe);
                    break;

                case Ingo_Rule_User::COMBINE_ANY:
                    /* Create filter if using OR */
                    $this->_addItem(
                        Ingo::RULE_FILTER,
                        new Ingo_Script_Procmail_Comment($rule->name, $rule->disable, true)
                    );
                    $loop = 0;
                    foreach ($rule->conditions as $condition) {
                        $recipe = new Ingo_Script_Procmail_Recipe(
                            array(
                                'action' => $class,
                                'action-value' => $rule->value,
                                'disable' => $rule->disable
                            ),
                            $this->_params
                        );
                        if ($loop++) {
                            $recipe->addFlag('E');
                        }
                        $recipe->addCondition($condition);
                        $this->_addItem(Ingo::RULE_FILTER, $recipe);
                    }
                    break;
                }
            }
        }

        // If an external delivery program is used, add final rule
        // to deliver to $DEFAULT
        if (isset($this->_params['delivery_agent'])) {
            $this->_addItem(
                Ingo::RULE_FILTER,
                new Ingo_Script_Procmail_Default($this->_params)
            );
        }
    }

    /**
     * Generates the procmail script to handle the blacklist specified in
     * the rules.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateBlacklist(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_BLACKLIST,
            new Ingo_Script_Procmail_Comment(_("Blacklisted Addresses"), $rule->disable, true)
        );

        $params = array(
            'action-value' => $rule->mailbox,
            'action' => strlen($rule->mailbox) ? 'Ingo_Rule_User_Move' : 'Ingo_Rule_User_Discard',
            'disable' => $rule->disable
        );

        foreach ($rule->addresses as $address) {
            $recipe = new Ingo_Script_Procmail_Recipe(
                $params,
                $this->_params
            );
            $recipe->addCondition(array(
                'field' => 'From',
                'value' => $address,
                'match' => 'address'
            ));
            $this->_addItem(Ingo::RULE_BLACKLIST, $recipe);
        }
    }

    /**
     * Generates the procmail script to handle the whitelist specified in
     * the rules.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateWhitelist(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_WHITELIST,
            new Ingo_Script_Procmail_Comment(_("Whitelisted Addresses"), $rule->disable, true)
        );

        foreach ($rule->addresses as $address) {
            $recipe = new Ingo_Script_Procmail_Recipe(array(
                'action' => 'Ingo_Rule_User_Keep',
                'disable' => $rule->disable
            ), $this->_params);
            $recipe->addCondition(array(
                'field' => 'From',
                'value' => $address,
                'match' => 'address'
            ));
            $this->_addItem(Ingo::RULE_WHITELIST, $recipe);
        }
    }

    /**
     * Generates the procmail script to handle vacation.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateVacation(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_VACATION,
            new Ingo_Script_Procmail_Comment(_("Vacation"), $rule->disable, true)
        );

        $recipe = new Ingo_Script_Procmail_Recipe(
            array(
                'action' => 'Ingo_Rule_System_Vacation',
                'action-value' => array(
                    'addresses' => $rule->addresses,
                    'subject' => $rule->subject,
                    'days' => $rule->days,
                    'reason' => $rule->reason,
                    'ignorelist' => $rule->ignore_list,
                    'excludes' => $rule->exclude,
                    'start' => $rule->start,
                    'end' => $rule->end
                ),
                'disable' => $rule->disable
            ),
            $this->_params
        );
        $this->_addItem(Ingo::RULE_VACATION, $recipe);
    }

    /**
     * Generates the procmail script to handle mail forwards.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateForward(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_FORWARD,
            new Ingo_Script_Procmail_Comment(_("Forwards"), $rule->disable, true)
        );

        $recipe = new Ingo_Script_Procmail_Recipe(
            array(
                'action' => 'Ingo_Rule_System_Forward',
                'action-value' => $rule->addresses,
                'disable' => $rule->disable
            ),
            $this->_params
        );

        if ($rule->keep) {
            $recipe->addFlag('c');
        }

        $this->_addItem(Ingo::RULE_FORWARD, $recipe);
    }

}
