<?php
/**
 * Copyright 2005-2007 Matt Weyland <mathias@weyland.ch>
 *
 * See the enclosed file LICENSE for license information (ASL).  If you
 * did not receive this file, see http://www.horde.org/licenses/apache.
 *
 * @author   Matt Weyland <mathias@weyland.ch>
 * @author   Jan Schneider <jan@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/apache ASL
 * @package  Ingo
 */

/**
 * The Ingo_Script_Maildrop class represents a maildrop script generator.
 *
 * @author   Matt Weyland <mathias@weyland.ch>
 * @author   Jan Schneider <jan@horde.org>
 * @category Horde
 * @license  http://www.horde.org/licenses/apache ASL
 * @package  Ingo
 */
class Ingo_Script_Maildrop extends Ingo_Script_Base
{
    /**
     * Additional storage action since maildrop does not support the "c-flag"
     * as in procmail.
     */
    const MAILDROP_STORAGE_ACTION_STOREANDFORWARD = 100;

    /**
     * A list of driver features.
     *
     * @var array
     */
    protected $_features = array(
        /* Can tests be case sensitive? */
        'case_sensitive' => true,
        /* Does the driver support setting IMAP flags? */
        'imap_flags' => false,
        /* Can this driver perform on demand filtering? */
        'on_demand' => false,
        /* Does the driver require a script file to be generated? */
        'script_file' => true,
        /* Does the driver support the stop-script option? */
        'stop_script' => false,
        /* Does the driver support vacation start and end on time level? */
        'vacation_time' => true,
    );

    /**
     * The list of actions allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_actions = array(
        'Ingo_Rule_User_Discard',
        'Ingo_Rule_User_Keep',
        'Ingo_Rule_User_Move',
        'Ingo_Rule_User_Redirect',
        'Ingo_Rule_User_RedirectKeep',
        'Ingo_Rule_User_Reject'
    );

    /**
     * The categories of filtering allowed.
     *
     * @var array
     */
    protected $_categories = array(
        'Ingo_Rule_System_Blacklist',
        'Ingo_Rule_System_Forward',
        'Ingo_Rule_System_Spam',
        'Ingo_Rule_System_Vacation',
        'Ingo_Rule_System_Whitelist'
    );

    /**
     * The types of tests allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_types = array(
        Ingo_Rule_User::TEST_HEADER
    );

    /**
     * The list of tests allowed (implemented) for this driver.
     *
     * @var array
     */
    protected $_tests = array(
        'contains', 'not contain',
        'is', 'not is',
        'begins with','not begins with',
        'ends with', 'not ends with',
        'regex', 'not regex',
        'matches', 'not matches',
        'exists', 'not exist',
        'less than', 'less than or equal to',
        'equal', 'not equal',
        'greater than', 'greater than or equal to',
    );

    /**
     * Generates the script to do the filtering specified in the rules.
     */
    protected function _generate()
    {
        $this->_addItem(
            Ingo::RULE_ALL,
            new Ingo_Script_Maildrop_Comment(_("maildrop script generated by Ingo") . ' (' . date('F j, Y, g:i a') . ')')
        );

        /* Add variable information, if present. */
        if (!empty($this->_params['variables']) &&
            is_array($this->_params['variables'])) {
            foreach ($this->_params['variables'] as $key => $val) {
                $this->_addItem(
                    Ingo::RULE_ALL,
                    new Ingo_Script_Maildrop_Variable(array('name' => $key, 'value' => $val))
                );
            }
        }

        $filters = Ingo_Storage_FilterIterator_Skip::create(
            $this->_params['storage'],
            $this->_params['skip']
        );

        foreach ($filters as $rule) {
            switch ($class = get_class($rule)) {
            case 'Ingo_Rule_System_Blacklist':
                $this->_generateBlacklist($rule);
                break;

            case 'Ingo_Rule_System_Forward':
                $this->_generateForward($rule);
                break;

            case 'Ingo_Rule_System_Spam':
                $this->_generateSpam($rule);
                break;

            case 'Ingo_Rule_System_Vacation':
                $this->_generateVacation($rule);
                break;

            case 'Ingo_Rule_System_Whitelist':
                $this->_generateWhitelist($rule);
                break;

            default:
                if (!in_array($class, $this->_actions)) {
                    break;
                }

                /* Create filter if using AND. */
                $recipe = new Ingo_Script_Maildrop_Recipe(
                    array(
                        'action' => $class,
                        'action-value' => $rule->value,
                        'combine' => $rule->combine,
                        'disable' => $rule->disable
                    ),
                    $this->_params
                );
                foreach ($rule->conditions as $condition) {
                    $recipe->addCondition($condition);
                }
                $this->_addItem(
                    Ingo::RULE_FILTER,
                    new Ingo_Script_Maildrop_Comment($rule->name, $rule->disable, true)
                );
                $this->_addItem(Ingo::RULE_FILTER, $recipe);
            }
        }
    }

    /**
     * Generates the maildrop script to handle the blacklist specified in
     * the rules.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateBlacklist(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_BLACKLIST,
            new Ingo_Script_Maildrop_Comment(_("Blacklisted Addresses"), $rule->disable, true)
        );

        $params = array(
            'action-value' => $rule->mailbox,
            'action' => strlen($rule->mailbox) ? 'Ingo_Rule_User_Move' : 'Ingo_Rule_User_Discard',
            'disable' => $rule->disable
        );

        foreach ($rule->addresses as $address) {
            $recipe = new Ingo_Script_Maildrop_Recipe($params, $this->_params);
            $recipe->addCondition(array(
                'field' => 'From',
                'value' => $address
            ));
            $this->_addItem(Ingo::RULE_BLACKLIST, $recipe);
        }
    }

    /**
     * Generates the maildrop script to handle the whitelist specified in
     * the rules.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateWhitelist(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_WHITELIST,
            new Ingo_Script_Maildrop_Comment(_("Whitelisted Addresses"), $rule->disable, true)
        );

        foreach ($rule->addresses as $address) {
            $recipe = new Ingo_Script_Maildrop_Recipe(
                array(
                    'action' => 'Ingo_Rule_User_Keep',
                    'disable' => $rule->disable
                ),
                $this->_params
            );
            $recipe->addCondition(array(
                'field' => 'From',
                'value' => $address
            ));
            $this->_addItem(Ingo::RULE_WHITELIST, $recipe);
        }
    }

    /**
     * Generates the maildrop script to handle mail forwards.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateForward(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_FORWARD,
            new Ingo_Script_Maildrop_Comment(_("Forwards"), $rule->disable, true)
        );

        $params = array(
            'action' => 'Ingo_Rule_System_Forward',
            'action-value' => $rule->addresses,
            'disable' => $rule->disable
        );
        if ($rule->keep) {
            $params['action'] = self::MAILDROP_STORAGE_ACTION_STOREANDFORWARD;
        }
        $recipe = new Ingo_Script_Maildrop_Recipe($params, $this->_params);
        $recipe->addCondition(array('field' => 'From', 'value' => ''));
        $this->_addItem(Ingo::RULE_FORWARD, $recipe);
    }

    /**
     * Generates the maildrop script to handle vacation messages.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateVacation(Ingo_Rule $rule)
    {
        if (!count($rule)) {
            return;
        }

        $this->_addItem(
            Ingo::RULE_VACATION,
            new Ingo_Script_Maildrop_Comment(_("Vacation"), $disable, true)
        );

        $recipe = new Ingo_Script_Maildrop_Recipe(
            array(
                'action' => 'Ingo_Rule_System_Vacation',
                'action-value' => array(
                    'addresses' => $rule->addresses,
                    'subject' => $rule->subject,
                    'days' => $rule->days,
                    'ignorelist' => $rule->ignore_list,
                    'excludes' => $rule->exclude,
                    'start' => $rule->start,
                    'end' => $rule->end
                ),
                'disable' => $disable
            ),
            $this->_params
        );

        $this->_addItem(Ingo::RULE_VACATION, $recipe);

        $this->_addItem(
            Ingo::RULE_VACATION,
            new Ingo_Script_String(
                Ingo_Rule_System_Vacation::vacationReason(
                    $rule->reason,
                    $rule->start,
                    $rule->end
                )
            ),
            'vacation.msg'
        );
    }

    /**
     * Generates the maildrop script to handle spam as identified by
     * SpamAssassin.
     *
     * @param Ingo_Rule $rule  Rule object.
     */
    protected function _generateSpam(Ingo_Rule $rule)
    {
        $this->_addItem(
            Ingo::RULE_SPAM,
            new Ingo_Script_Maildrop_Comment(_("Spam Filter"), $rule->disable, true)
        );

        $recipe = new Ingo_Script_Maildrop_Recipe(
            array(
                'action-value' => $rule->mailbox,
                'action' => strlen($rule->mailbox) ? 'Ingo_Rule_User_Move' : 'Ingo_Rule_User_Discard',
                'disable' => $rule->disable
            ),
            $this->_params
        );

        if ($this->_params['spam_compare'] == 'numeric') {
            $recipe->addCondition(array(
                'match' => 'greater than or equal to',
                'field' => $this->_params['spam_header'],
                'value' => $rule->level
            ));
        } elseif ($this->_params['spam_compare'] == 'string') {
            $recipe->addCondition(array(
                'match' => 'contains',
                'field' => $this->_params['spam_header'],
                'value' => str_repeat(
                    $this->_params['spam_char'],
                    $rule->level
                )
            ));
        } elseif ($this->_params['spam_compare'] == 'boolean') {
            $recipe->addCondition(array(
                'match' => 'is',
                'field' => $this->_params['spam_header'],
                'value' => $this->_params['spam_value']
            ));
        }

        $this->_addItem(Ingo::RULE_SPAM, $recipe);
    }

}
