<?php
/**
 * Horde_Vcs_Git implementation.
 *
 * Constructor args:
 * <pre>
 * 'sourceroot': The source root for this repository
 * 'paths': Hash with the locations of all necessary binaries: 'git'
 * </pre>
 *
 * @TODO find bad output earlier - use proc_open, check stderr or result codes?
 *
 * Copyright 2008-2012 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.horde.org/licenses/lgpl21.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Michael Slusarz <slusarz@horde.org>
 * @package Vcs
 */
class Horde_Vcs_Git extends Horde_Vcs_Base
{
    /**
     * The current driver.
     *
     * @var string
     */
    protected $_driver = 'Git';

    /**
     * Driver features.
     *
     * @var array
     */
    protected $_features = array(
        'deleted'   => false,
        'patchsets' => true,
        'branches'  => true,
        'snapshots' => true);

    /**
     * The available diff types.
     *
     * @var array
     */
    protected $_diffTypes = array('unified');

    /**
     * The list of branches for the repo.
     *
     * @var array
     */
    protected $_branchlist;

    /**
     * The git version
     *
     * @var string
     */
    public $version;

    /**
     * @throws Horde_Vcs_Exception
     */
    public function __construct($params = array())
    {
        parent::__construct($params);

        if (!is_executable($this->getPath('git'))) {
            throw new Horde_Vcs_Exception('Missing git binary (' . $this->getPath('git') . ' is missing or not executable)');
        }

        $v = trim(shell_exec($this->getPath('git') . ' --version'));
        $this->version = preg_replace('/[^\d\.]/', '', $v);

        // Try to find the repository if we don't have the exact path. @TODO put
        // this into a builder method/object and cache the results.
        if (!file_exists($this->sourceroot . '/HEAD')) {
            if (file_exists($this->sourceroot . '.git/HEAD')) {
                $this->_sourceroot .= '.git';
            } elseif (file_exists($this->sourceroot . '/.git/HEAD')) {
                $this->_sourceroot .= '/.git';
            } else {
                throw new Horde_Vcs_Exception('Can not find git repository.');
            }
        }
    }

    /**
     * TODO
     */
    public function isValidRevision($rev)
    {
        return $rev && preg_match('/^[a-f0-9]+$/i', $rev);
    }

    /**
     * TODO
     */
    public function isFile($where, $branch = null)
    {
        if (!$branch) {
            $branch = $this->getDefaultBranch();
        }

        $where = str_replace($this->sourceroot . '/', '', $where);
        $command = $this->getCommand() . ' ls-tree ' . escapeshellarg($branch) . ' ' . escapeshellarg($where) . ' 2>&1';
        exec($command, $entry, $retval);

        if (!count($entry)) { return false; }

        $data = explode(' ', $entry[0]);
        return ($data[1] == 'blob');
    }

    /**
     * TODO
     */
    public function getCommand()
    {
        return escapeshellcmd($this->getPath('git'))
            . ' --git-dir=' . escapeshellarg($this->sourceroot);
    }

    /**
     * Runs a git commands.
     *
     * Uses proc_open() to properly catch errors and returns a stream with the
     * command result. fclose() must be called manually on the returned stream
     * and proc_close() on the resource, once the output stream has been
     * finished reading.
     *
     * @param string $args  Any arguments for the git command. Must be escaped.
     *
     * @return array(resource, stream)  The process resource and the command
     *                                  output.
     * @throws Horde_Vcs_Exception if command cannot be executed or returns an
     *                             error.
     */
    public function runCommand($args)
    {
        $cmd = $this->getCommand() . ' ' . $args;
        $stream = proc_open(
            $cmd,
            array(1 => array('pipe', 'w'), 2 => array('pipe', 'w')),
            $pipes);
        if (!$stream || !is_resource($stream)) {
            throw new Horde_Vcs_Exception('Failed to execute git: ' . $cmd);
        }
        stream_set_blocking($pipes[2], 0);
        if ($error = stream_get_contents($pipes[2])) {
            fclose($pipes[2]);
            proc_close($stream);
            throw new Horde_Vcs_Exception($error);
        }
        fclose($pipes[2]);
        return array($stream, $pipes[1]);
    }

    /**
     * TODO
     *
     * @throws Horde_Vcs_Exception
     */
    public function annotate($fileob, $rev)
    {
        $this->assertValidRevision($rev);

        $command = $this->getCommand() . ' blame -p ' . escapeshellarg($rev) . ' -- ' . escapeshellarg($fileob->getSourcerootPath()) . ' 2>&1';
        $pipe = popen($command, 'r');
        if (!$pipe) {
            throw new Horde_Vcs_Exception('Failed to execute git annotate: ' . $command);
        }

        $curr_rev = null;
        $db = $lines = array();
        $lines_group = $line_num = 0;

        while (!feof($pipe)) {
            $line = rtrim(fgets($pipe, 4096));

            if (!$line || ($line[0] == "\t")) {
                if ($lines_group) {
                    $lines[] = array(
                        'author' => $db[$curr_rev]['author'] . ' ' . $db[$curr_rev]['author-mail'],
                        'date' => $db[$curr_rev]['author-time'],
                        'line' => $line ? substr($line, 1) : '',
                        'lineno' => $line_num++,
                        'rev' => $curr_rev
                    );
                    --$lines_group;
                }
            } elseif ($line != 'boundary') {
                if ($lines_group) {
                    list($prefix, $linedata) = explode(' ', $line, 2);
                    switch ($prefix) {
                    case 'author':
                    case 'author-mail':
                    case 'author-time':
                    //case 'author-tz':
                        $db[$curr_rev][$prefix] = trim($linedata);
                        break;
                    }
                } else {
                    $curr_line = explode(' ', $line);
                    $curr_rev = $curr_line[0];
                    $line_num = $curr_line[2];
                    $lines_group = isset($curr_line[3]) ? $curr_line[3] : 1;
                }
            }
        }

        pclose($pipe);

        return $lines;
    }

    /**
     * Function which returns a file pointing to the head of the requested
     * revision of a file.
     *
     * @param string $fullname  Fully qualified pathname of the desired file
     *                          to checkout
     * @param string $rev       Revision number to check out
     *
     * @return resource  A stream pointer to the head of the checkout.
     * @throws Horde_Vcs_Exception
     */
    public function checkout($file, $rev)
    {
        $this->assertValidRevision($rev);

        $file_ob = $this->getFile($file);
        $hash = $file_ob->getHashForRevision($rev);
        if ($hash == '0000000000000000000000000000000000000000') {
            throw new Horde_Vcs_Exception($file . ' is deleted in commit ' . $rev);
        }

        if ($pipe = popen($this->getCommand() . ' cat-file blob ' . $hash . ' 2>&1', VC_WINDOWS ? 'rb' : 'r')) {
            return $pipe;
        }

        throw new Horde_Vcs_Exception('Couldn\'t perform checkout of the requested file');
    }

    /**
     * Create a range of revisions between two revision numbers.
     *
     * @param Horde_Vcs_File_Git $file  The desired file.
     * @param string $r1                The initial revision.
     * @param string $r2                The ending revision.
     *
     * @return array  The revision range, or empty if there is no straight
     *                line path between the revisions.
     */
    public function getRevisionRange(Horde_Vcs_File_Base $file, $r1, $r2)
    {
        $revs = $this->_getRevisionRange($file, $r1, $r2);
        return empty($revs)
            ? array_reverse($this->_getRevisionRange($file, $r2, $r1))
            : $revs;
    }

    /**
     * TODO
     */
    protected function _getRevisionRange(Horde_Vcs_File_Git $file, $r1, $r2)
    {
        $cmd = $this->getCommand() . ' rev-list ' . escapeshellarg($r1 . '..' . $r2) . ' -- ' . escapeshellarg($file->getSourcerootPath());
        $revs = array();

        exec($cmd, $revs);
        return array_map('trim', $revs);
    }

    /**
     * Obtain the differences between two revisions of a file.
     *
     * @param Horde_Vcs_File_Git $file  The desired file.
     * @param string $rev1              Original revision number to compare
     *                                  from.
     * @param string $rev2              New revision number to compare against.
     * @param array $opts               The following optional options:
     *                                  - 'num': (integer) DEFAULT: 3
     *                                  - 'type': (string) DEFAULT: 'unified'
     *                                  - 'ws': (boolean) DEFAULT: true
     *
     * @return string  The diff text.
     */
    protected function _diff(Horde_Vcs_File_Base $file, $rev1, $rev2, $opts)
    {
        $diff = array();
        $flags = '';

        if (!$opts['ws']) {
            $flags .= ' -b -w ';
        }

        if (!$rev1) {
            $command = $this->getCommand() . ' show --oneline ' . escapeshellarg($rev2) . ' -- ' . escapeshellarg($file->getSourcerootPath()) . ' 2>&1';
        } else {
            switch ($opts['type']) {
            case 'unified':
                $flags .= '--unified=' . escapeshellarg((int)$opts['num']);
                break;
            }

            // @TODO: add options for $hr options - however these may not
            // be compatible with some diffs.
            $command = $this->getCommand() . ' diff -M -C ' . $flags . ' --no-color ' . escapeshellarg($rev1 . '..' . $rev2) . ' -- ' . escapeshellarg($file->getSourcerootPath()) . ' 2>&1';
        }

        exec($command, $diff, $retval);
        return $diff;
    }

    /**
     * Returns an abbreviated form of the revision, for display.
     *
     * @param string $rev  The revision string.
     *
     * @return string  The abbreviated string.
     */
    public function abbrev($rev)
    {
        return substr($rev, 0, 7) . '[...]';
    }

    /**
     * TODO
     */
    public function getBranchList()
    {
        if (!isset($this->_branchlist)) {
            $this->_branchlist = array();
            exec($this->getCommand() . ' show-ref --heads', $branch_list);

            foreach ($branch_list as $val) {
                $line = explode(' ', trim($val), 2);
                $this->_branchlist[substr($line[1], strrpos($line[1], '/') + 1)] = $line[0];
            }
        }

        return $this->_branchlist;
    }

    /**
     * @TODO ?
     */
    public function getDefaultBranch()
    {
        return 'master';
    }
}
